/*

License $Id: AbstractContentHandlerImpl.java,v 1.1 2001/08/06 16:46:42 Hendrik Exp $

Copyright (c) 2001 tagtraum industries.

The sixbs library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

For LGPL see <http://www.fsf.org/copyleft/lesser.txt>

We encourage your feedback and suggestions and want to use your feedback to
improve the Software. Send all such feedback to:
feedback@tagtraum.com

For more information on tagtraum industries, Hendrik Schreiber and sixbs
please see <http://www.tagtraum.com/>.

*/

package com.tagtraum.sixbs;

import java.util.*;
import java.io.*;
import java.lang.reflect.*;
import java.net.URL;

import org.xml.sax.*;
import org.xml.sax.helpers.*;

import com.tagtraum.framework.util.*;

/**
 * Is notified by the parser and build the objects from the notifications.
 * <p>Uses SAX2, therefore part of the documentation
 * is from the SAX2 docs.
 * 
 * @author 	Hendrik Schreiber
 * @version $Id: AbstractContentHandlerImpl.java,v 1.1 2001/08/06 16:46:42 Hendrik Exp $
 */
abstract class AbstractContentHandlerImpl implements ContentHandler {

    /**
     * Source-Version
     */
    public static String	vcid = 
	"$Id: AbstractContentHandlerImpl.java,v 1.1 2001/08/06 16:46:42 Hendrik Exp $";
    
    protected static final Class[] STRING_OBJECT = new Class[] {String.class};
    protected static final Null NULL = Null.UNIQUE_NULL;
    protected static final String TYPED_ARRAY = TypedArray.class.getName();
    protected static final String SIXBS = "sixbs";
    
    protected static ResourceBundle localStrings = ResourceBundle.getBundle("com.tagtraum.sixbs.localStrings");
    
    protected Object[] ONEARG = new Object[] {null};
    protected static final int OBJECT = 0;
    protected static final int PROPERTY = 1;
    protected static final int ARRAY = 2;
    protected static final int LITERAL = 3;

    protected InputSource inputSource;
    protected ObjectIdentifier objectIdentifier;
    protected ArrayList stack;
    protected StringBuffer cdata;
    protected SIXBSReader reader;

    public AbstractContentHandlerImpl(SIXBSReader reader) {
        this.reader = reader;
        objectIdentifier = new ObjectIdentifier();
        cdata = new StringBuffer(256);
        stack = new ArrayList();
    }

    /**
     * Receive notification of the end of an element.
     *
     * <p>The SAX parser will invoke this method at the end of every
     * element in the XML document; there will be a corresponding
     * {@link #startElement startElement} event for every endElement
     * event (even when the element is empty).</p>
     *
     * <p>For information on the names, see startElement.</p>
     *
     * @param uri The Namespace URI, or the empty string if the
     *        element has no Namespace URI or if Namespace
     *        processing is not being performed.
     * @param localName The local name (without prefix), or the
     *        empty string if Namespace processing is not being
     *        performed.
     * @param qName The qualified XML 1.0 name (with prefix), or the
     *        empty string if qualified names are not available.
     * @exception org.xml.sax.SAXException Any SAX exception, possibly
     *            wrapping another exception.
     */
    public void endElement (String namespaceURI, String localName,
                String qName) throws SAXException {
        if (empty() && localName.equals(SIXBS)) return;
        State state = pop();
        //System.out.println("</" + localName + ">: " + state);
        try {
            makeObject(state);
            if (state.getState() == OBJECT) {
                ONEARG[0] = getStateContextObject(state);
                if (state.getId() != null) objectIdentifier.registerObject(state.getId(), ONEARG[0]);
                if (!empty()) {
                    State substate = peek();
                    switch (substate.getState()) {
                        case PROPERTY:
                            if (depth() > 1) {
                                Object object = peek(2).getContext();
                                //if (state.getContext() instanceof Null) ONEARG[0]=null;
                                Method method = (Method)substate.getContext();

/*
                                System.out.println("substate: " + substate);
                                System.out.println("localName: " + localName);
                                System.out.println("method: " + method);
                                System.out.println("object: " + object);
*/

                                method.invoke(object, ONEARG);
                            }
                            else {
                                throw new SAXException(localStrings.getString("unexpected_property") + " " + substate.getContext());
                            }
                            break;
                        case ARRAY:
                            //System.out.println("adding to typed array: " + ONEARG[0]);
                            ((TypedArray)substate.getContext()).add(ONEARG[0]);
                            break;
                        default:
                            throw new SAXException(localStrings.getString("unexpected_state") + " " + substate);
                    }
                }
                else {
                    reader.addObject(ONEARG[0]);
                }
            }
        }
        catch (InstantiationException ie) {
            throw new SAXException(localStrings.getString("instantiation_failed") + " " + ie);
        }
        catch (IllegalAccessException iae) {
            throw new SAXException(localStrings.getString("method_invocation_forbidden") + " " + iae);
        }
        catch (InvocationTargetException ite) {
            throw new SAXException(localStrings.getString("property_write_invocation_failed") + " " + ite.getTargetException());
        }
        catch (SIXBSException sixbse) {
            throw new SAXException(sixbse);
        }
        catch (RuntimeException rte) {
            rte.printStackTrace();
            throw new SAXException(rte);
        }
    }

    /**
     * Returns the object that is held in the state's context in its needed form.
     */
    protected Object getStateContextObject(State state) throws SIXBSException {
        if (state.getContext() instanceof Null) {
            return null;
        }
        else if (state.getContext() instanceof Adapter) {
            return ((Adapter)state.getContext()).deserialize();
        }
        else {
            return state.getContext();
        }
    }

    /**
     * Transforms TypedArrays and Literals into real objects and replaces them on the stack.
     */
    protected void makeObject(State state) throws InstantiationException, IllegalAccessException, InvocationTargetException {
        switch (state.getState()) {
            case ARRAY:
                Object array = ((TypedArray)state.getContext()).toTypedArray();
                state.set(OBJECT, array);
                return;
            case LITERAL:
                ONEARG[0] = decode(cdata.toString());
                cdata.setLength(0);
                Object literal = ((Constructor)state.getContext()).newInstance(ONEARG);
                state.set(OBJECT, literal);
                return;
            default:
                return;
        }
    }

    /**
     * Receive notification of character data.
     *
     * <p>The Parser will call this method to report each chunk of
     * character data.  SAX parsers may return all contiguous character
     * data in a single chunk, or they may split it into several
     * chunks; however, all of the characters in any single event
     * must come from the same external entity so that the Locator
     * provides useful information.</p>
     *
     * <p>The application must not attempt to read from the array
     * outside of the specified range.</p>
     *
     * <p>Note that some parsers will report whitespace in element
     * content using the {@link #ignorableWhitespace ignorableWhitespace}
     * method rather than this one (validating parsers <em>must</em> 
     * do so).</p>
     *
     * @param ch The characters from the XML document.
     * @param start The start position in the array.
     * @param length The number of characters to read from the array.
     * @exception org.xml.sax.SAXException Any SAX exception, possibly
     *            wrapping another exception.
     * @see #ignorableWhitespace 
     * @see org.xml.sax.Locator
     */
    public void characters (char ch[], int start, int length)throws SAXException {
        cdata.append(ch, start, length);
    }


    /**
     * Receive an object for locating the origin of SAX document events.
     *
     * <p>SAX parsers are strongly encouraged (though not absolutely
     * required) to supply a locator: if it does so, it must supply
     * the locator to the application by invoking this method before
     * invoking any of the other methods in the ContentHandler
     * interface.</p>
     *
     * <p>The locator allows the application to determine the end
     * position of any document-related event, even if the parser is
     * not reporting an error.  Typically, the application will
     * use this information for reporting its own errors (such as
     * character content that does not match an application's
     * business rules).  The information returned by the locator
     * is probably not sufficient for use with a search engine.</p>
     *
     * <p>Note that the locator will return correct information only
     * during the invocation of the events in this interface.  The
     * application should not attempt to use it at any other time.</p>
     *
     * @param locator An object that can return the location of
     *                any SAX document event.
     * @see org.xml.sax.Locator
     */
    public void setDocumentLocator (Locator locator) {
    }


    /**
     * Receive notification of the beginning of a document.
     *
     * <p>The SAX parser will invoke this method only once, before any
     * other methods in this interface or in {@link org.xml.sax.DTDHandler
     * DTDHandler} (except for {@link #setDocumentLocator 
     * setDocumentLocator}).</p>
     *
     * @exception org.xml.sax.SAXException Any SAX exception, possibly
     *            wrapping another exception.
     * @see #endDocument
     */
    public void startDocument() throws SAXException {
    }


    /**
     * Receive notification of the end of a document.
     *
     * <p>The SAX parser will invoke this method only once, and it will
     * be the last method invoked during the parse.  The parser shall
     * not invoke this method until it has either abandoned parsing
     * (because of an unrecoverable error) or reached the end of
     * input.</p>
     *
     * @exception org.xml.sax.SAXException Any SAX exception, possibly
     *            wrapping another exception.
     * @see #startDocument
     */
    public void endDocument() throws SAXException {
        //System.out.println("EndDocument: " + empty());
        //if (state == OBJECT) throw new SAXException("SIXBSReader: Format error.");
        reader.setStillParsing(false);
    }

    /**
     * Begin the scope of a prefix-URI Namespace mapping.
     *
     * <p>The information from this event is not necessary for
     * normal Namespace processing: the SAX XML reader will 
     * automatically replace prefixes for element and attribute
     * names when the <code>http://xml.org/sax/features/namespaces</code>
     * feature is <var>true</var> (the default).</p>
     *
     * <p>There are cases, however, when applications need to
     * use prefixes in character data or in attribute values,
     * where they cannot safely be expanded automatically; the
     * start/endPrefixMapping event supplies the information
     * to the application to expand prefixes in those contexts
     * itself, if necessary.</p>
     *
     * <p>Note that start/endPrefixMapping events are not
     * guaranteed to be properly nested relative to each-other:
     * all startPrefixMapping events will occur before the
     * corresponding {@link #startElement startElement} event, 
     * and all {@link #endPrefixMapping endPrefixMapping}
     * events will occur after the corresponding {@link #endElement
     * endElement} event, but their order is not otherwise 
     * guaranteed.</p>
     *
     * <p>There should never be start/endPrefixMapping events for the
     * "xml" prefix, since it is predeclared and immutable.</p>
     *
     * @param prefix The Namespace prefix being declared.
     * @param uri The Namespace URI the prefix is mapped to.
     * @exception org.xml.sax.SAXException The client may throw
     *            an exception during processing.
     * @see #endPrefixMapping
     * @see #startElement
     */
    public void startPrefixMapping (String prefix, String uri) throws SAXException {
    }


    /**
     * End the scope of a prefix-URI mapping.
     *
     * <p>See {@link #startPrefixMapping startPrefixMapping} for 
     * details.  This event will always occur after the corresponding 
     * {@link #endElement endElement} event, but the order of 
     * {@link #endPrefixMapping endPrefixMapping} events is not otherwise
     * guaranteed.</p>
     *
     * @param prefix The prefix that was being mapping.
     * @exception org.xml.sax.SAXException The client may throw
     *            an exception during processing.
     * @see #startPrefixMapping
     * @see #endElement
     */
    public void endPrefixMapping (String prefix) throws SAXException {
    }


    /**
     * Receive notification of ignorable whitespace in element content.
     *
     * <p>Validating Parsers must use this method to report each chunk
     * of whitespace in element content (see the W3C XML 1.0 recommendation,
     * section 2.10): non-validating parsers may also use this method
     * if they are capable of parsing and using content models.</p>
     *
     * <p>SAX parsers may return all contiguous whitespace in a single
     * chunk, or they may split it into several chunks; however, all of
     * the characters in any single event must come from the same
     * external entity, so that the Locator provides useful
     * information.</p>
     *
     * <p>The application must not attempt to read from the array
     * outside of the specified range.</p>
     *
     * @param ch The characters from the XML document.
     * @param start The start position in the array.
     * @param length The number of characters to read from the array.
     * @exception org.xml.sax.SAXException Any SAX exception, possibly
     *            wrapping another exception.
     * @see #characters
     */
    public void ignorableWhitespace (char ch[], int start, int length) throws SAXException {
    }


    /**
     * Receive notification of a processing instruction.
     *
     * <p>The Parser will invoke this method once for each processing
     * instruction found: note that processing instructions may occur
     * before or after the main document element.</p>
     *
     * <p>A SAX parser must never report an XML declaration (XML 1.0,
     * section 2.8) or a text declaration (XML 1.0, section 4.3.1)
     * using this method.</p>
     *
     * @param target The processing instruction target.
     * @param data The processing instruction data, or null if
     *        none was supplied.  The data does not include any
     *        whitespace separating it from the target.
     * @exception org.xml.sax.SAXException Any SAX exception, possibly
     *            wrapping another exception.
     */
    public void processingInstruction (String target, String data) throws SAXException {
    }


    /**
     * Receive notification of a skipped entity.
     *
     * <p>The Parser will invoke this method once for each entity
     * skipped.  Non-validating processors may skip entities if they
     * have not seen the declarations (because, for example, the
     * entity was declared in an external DTD subset).  All processors
     * may skip external entities, depending on the values of the
     * <code>http://xml.org/sax/features/external-general-entities</code>
     * and the
     * <code>http://xml.org/sax/features/external-parameter-entities</code>
     * properties.</p>
     *
     * @param name The name of the skipped entity.  If it is a 
     *        parameter entity, the name will begin with '%', and if
     *        it is the external DTD subset, it will be the string
     *        "[dtd]".
     * @exception org.xml.sax.SAXException Any SAX exception, possibly
     *            wrapping another exception.
     */
    public void skippedEntity (String name) throws SAXException {
    }
    

    protected State peek() {
        return (State)stack.get(stack.size()-1);
    }

    protected int depth() {
        return stack.size();
    }

    protected State peek(int depth) {
        return (State)stack.get(stack.size()-depth);
    }
    
    protected State pop() {
        return (State)stack.remove(stack.size()-1);
    }
    
    protected void push(State object) {
        stack.add(object);
    }

    protected boolean empty() {
        return stack.isEmpty();
    }
    
    public final static String decode(String name) {
        //System.out.println("DECODE: " + name);
        return name;

        // apparently we don't need decoding

        /*
        StringHelper.
        if (name.equals("lt")) {
            return "<";
        } else if (name.equals("gt")) {
            return ">";
        } else if (name.equals("amp")) {
            return "&";
        } else if (name.equals("quot")) {
            return "\"";
        } else if (name.equals("apos")) {
            return "'";
        } else if (name.startsWith("#x")) {
            char[] cbuf = new char[1];
    
            cbuf[0] = (char) Integer.parseInt(name.substring(1), 16);
    
            return new String(cbuf);
        } else if (name.startsWith("#")) {
            char[] cbuf = new char[1];
    
            cbuf[0] = (char) Integer.parseInt(name.substring(1));
    
            return new String(cbuf);
        }
        */
    }
}