/*

License $Id: AbstractSIXBSWriterImpl.java,v 1.1 2001/08/06 16:46:42 Hendrik Exp $

Copyright (c) 2001 tagtraum industries.

The sixbs library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

For LGPL see <http://www.fsf.org/copyleft/lesser.txt>

We encourage your feedback and suggestions and want to use your feedback to
improve the Software. Send all such feedback to:
feedback@tagtraum.com

For more information on tagtraum industries, Hendrik Schreiber and sixbs
please see <http://www.tagtraum.com/>.

*/

package com.tagtraum.sixbs;

import org.w3c.dom.*;
import java.util.*;
import java.io.*;
import java.net.URL;
import java.lang.reflect.*;
import com.tagtraum.framework.util.*;



/**
 * Base class for SIXBSWriterHandlers.
 *
 * @author 	Hendrik Schreiber
 * @version $Id: AbstractSIXBSWriterImpl.java,v 1.1 2001/08/06 16:46:42 Hendrik Exp $
 */
abstract class AbstractSIXBSWriterImpl {

    /**
     * Source-Version
     */
    public static String	vcid =
	"$Id: AbstractSIXBSWriterImpl.java,v 1.1 2001/08/06 16:46:42 Hendrik Exp $";

    public static final Null NULL = Null.UNIQUE_NULL;
    public static final String REFERENCE = Reference.class.getName();
    public static ResourceBundle localStrings = ResourceBundle.getBundle("com.tagtraum.sixbs.localStrings");
    public static final Object[] NOARGS = new Object[] {};

    private Object[] ONEARG = new Object[] {null};
    private Class[] ONECLASS = new Class[] {null};
    private ObjectIdentifier objectIdentifier;
    private String encoding;
    private boolean append;
    private AdapterFinder adapterFinder;

    protected Writer out;

    /**
     * For performance reasons the Writer should be buffered. sixbs does <i>not</i>
     * buffer. sixbs will write with the systems default encoding or the encoding
     * set in the provided buffer.
     *
     * @param out Writer to write objects to with the specified encoding.
     * @param append if true, no XML declaration (&lt;?xml version="1.0"?&gt;)is written.
     * @param adapterFinder the adapterFinder to use
     * @param encoding encoding that is used by the writer - set null, if unknwon
     */
    public AbstractSIXBSWriterImpl(Writer out, AdapterFinder adapterFinder, boolean append, String encoding) throws IOException {
        this.out = out;
        objectIdentifier = new ObjectIdentifier();
        this.adapterFinder = adapterFinder;
        this.encoding = encoding;
        this.append = append;
        if (!append) {
            writeHeader();
        }
    }

    /**
     * Writes an Object.
     */
    public synchronized void writeObject(Object obj) throws IOException {
        if (obj == null) obj = NULL;
        String id = null;
        if ((id = objectIdentifier.getId(obj)) != null) {
            writeReference(id);
        }
        else if (obj.getClass().isArray()) {
            writeArray(obj, objectIdentifier.createId(obj));
        }
        else {
            Class adapterClass = null;
            if (obj instanceof Adapter || (adapterClass = adapterFinder.find(obj)) == null) {
                writePlainObject(obj, objectIdentifier.createId(obj));
            }
            else {
                writeAdapter(obj, adapterClass);
            }
        }
    }

    /**
     * Writes a Reference to an object that has been written before.
     */
    abstract public void writeReference(String id) throws IOException;

    /**
     * Writes an Array. It is garanteed that this Array has not been written before.
     */
    abstract public void writeArray(Object obj, String id) throws IOException;

    /**
     * Writes a plain Object. It is garanteed that this Object has not been written before.
     */
    abstract public void writePlainObject(Object obj, String id) throws IOException;

    /**
     * Returns the version of this WriterHandler, e.g. "1.0".
     */
    abstract public String getVersion();

    /**
     * Writes an Adapter.
     */
    private void writeAdapter(Object obj, Class adapterClass) throws IOException {
        Object adapter = null;
        try {
            ONECLASS[0] = obj.getClass();
            Constructor constr = adapterClass.getConstructor(ONECLASS);
            ONEARG[0]=obj;
            adapter = constr.newInstance(ONEARG);
        }
        catch (Throwable t) {
            throw new SIXBSException(localStrings.getString("adapter_instantiation_failure") + " " + adapterClass.getName(), t);
        }
        writeObject(adapter);
    }

    /**
     * Writes the name of a class. Helpermethod.
     */
    public void writeClass(Object obj) throws IOException {
        out.write(obj.getClass().getName());
    }

    /**
     * Inidcates whether a property with a given name should be written or not.
     */
    public boolean writeProperty(Object obj, String propertyName) {
        if (obj instanceof SIXBSSerializable) {
            return ((SIXBSSerializable)obj).serializable(propertyName);
        }
        return true;
    }

    /**
     * Writes the xml declaration, and an opening &lt;sixbs&gt; tag. Is called, if
     * append to false.
     */
    private void writeHeader() throws IOException {
        if (encoding == null) {
            out.write("<?xml version=\"1.0\"?>");
        }
        else {
            out.write("<?xml version=\"1.0\" encoding=\"");
            out.write(encoding);
            out.write("\" ?>");
        }
        writeXMLDeclarations();
        out.write("<sixbs version=\""+getVersion()+"\">");
    }

    /**
     * Writes additional xml declarations like entities and doctype.
     * Default implementation is empty.
     */
    protected void writeXMLDeclarations() throws IOException {
    }

    /**
     * Writes a closing &lt;/sixbs&gt; tag. Is called, if
     * append to false.
     */
    private void writeFooter() throws IOException {
        out.write("</sixbs>");
    }

    /**
     * Flush the stream.  If the stream has saved any characters from the
     * various write() methods in a buffer, write them immediately to their
     * intended destination.  Then, if that destination is another character or
     * byte stream, flush it.  Thus one flush() invocation will flush all the
     * buffers in a chain of Writers and OutputStreams.
     *
     * @exception  IOException  If an I/O error occurs
     */
    public void flush() throws IOException {
        out.flush();
    }

    /**
     * Close the stream, flushing it first.  Once a stream has been closed,
     * further write() or flush() invocations will cause an IOException to be
     * thrown.
     * If this writer does not operate in append-mode, a closing &lt;/sixbs&gt;
     * tag will be written.
     *
     * @exception  IOException  If an I/O error occurs
     * @see #writeHeader()
     * @see #writeFooter()
     */
    public void close() throws IOException {
        if (!append) {
            writeFooter();
        }
        objectIdentifier = null;
        out.close();
    }

    /**
     * Encodes the string value of an element (text node) to an
     * XML conform format, special characters such the ampersand (&)
     * are replaced by an entities (&amp;).
     *
     * @param elementValue the string value to be encoded
     * @return the XML compliant string value.
     */
    public static String encode(String elementValue) {
        StringBuffer str = new StringBuffer();
        int len = (elementValue != null ? elementValue.length() : 0);
        char[] cbuf = elementValue.toCharArray();

        for (int i = 0; i < len; i++) {
            switch (cbuf[i]) {

            case '<': {
            str.append("&lt;");

            break;
            }

            case '>': {
            str.append("&gt;");

            break;
            }

            case '&': {
            str.append("&amp;");

            break;
            }

            case '\'': {
            str.append("&apos;");

            break;
            }

            case '"': {
            str.append("&quot;");

            break;
            }

            default: {
            if ((int)cbuf[i] < 32 || (int)cbuf[i] > 127) {
                str.append("&#");
                str.append((int)cbuf[i]);
                str.append(';');
            }
            else {
                str.append(cbuf[i]);
            }
            }
            }
        }
    return str.toString();
    }

}