/*

License $Id: SIXBSReader.java,v 1.5 2001/08/06 16:46:42 Hendrik Exp $

Copyright (c) 2001 tagtraum industries.

The sixbs library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

For LGPL see <http://www.fsf.org/copyleft/lesser.txt>

We encourage your feedback and suggestions and want to use your feedback to
improve the Software. Send all such feedback to:
feedback@tagtraum.com

For more information on tagtraum industries, Hendrik Schreiber and sixbs
please see <http://www.tagtraum.com/>.

*/

package com.tagtraum.sixbs;

import java.util.*;
import java.io.*;
import java.net.URL;
import javax.xml.parsers.*;
import org.xml.sax.*;
import com.tagtraum.framework.util.EntityResolverImpl;

/**
 * DeSerializes beans from XML. Objects are read asynchronously therefore
 * excellent performance is ensured. How many objects are read ahead, can
 * be specified - you should do this, if you are intending to read many,
 * or very large objects. Also, if speed is very important to you, you
 * should buffer the InputStream or Reader (if this is not done by your SAX2
 * parser) and set readAhead to a high value.
 *
 * @author 	<a href="mailto:hs@tagtraum.com">Hendrik Schreiber</a>
 * @version $Id: SIXBSReader.java,v 1.5 2001/08/06 16:46:42 Hendrik Exp $
 * @see SIXBSWriter
 */
public class SIXBSReader {

    /**
     * Source-Version
     */
    public static String	vcid =
	"$Id: SIXBSReader.java,v 1.5 2001/08/06 16:46:42 Hendrik Exp $";

    //private static ResourceBundle localStrings = ResourceBundle.getBundle("com.tagtraum.sixbs.localStrings");

    private static SAXParserFactory saxParserFactory;
    public static String DEFAULTVERSION = "1.1";

    private boolean stillParsing = true;
    private ParseThread parseThread;
    private IOException pendingException;
    private InputSource inputSource;
    private ObjectIdentifier objectIdentifier;
    private List objects = new ArrayList();
    private Object lock = new Object();
    private int readAhead = 10;
    private String defaultVersion;
    private String version;
    private boolean validating = false;

    public SIXBSReader(Reader in) throws IOException {
        this(in, DEFAULTVERSION);
    }

    public SIXBSReader(InputStream in) throws IOException {
        this(in, DEFAULTVERSION);
    }

    /**
     * Deserializes from an XML document using the given sixbs version if
     * no version is specified in the document itself.
     */
    public SIXBSReader(Reader in, String defaultVersion) throws IOException {
        inputSource = new InputSource(in);
        init(defaultVersion);
    }

    /**
     * Deserializes from an XML document using the given sixbs version if
     * no version is specified in the document itself.
     */
    public SIXBSReader(InputStream in, String defaultVersion) throws IOException {
        inputSource = new InputSource(in);
        init(defaultVersion);
    }

    private void init(String defaultVersion) {
        objectIdentifier = new ObjectIdentifier();
        this.defaultVersion = defaultVersion;
    }

    /**
     * Indicates whether validating is turned on.
     *
     * @see #setValidating(boolean)
     */
    public boolean getValidating() {
        return validating;
    }

    /**
     * Turns on validating. This makes only sense, if you wrote a
     * <i>valid</i> xml document before, i.e. you didn't use the <code>append</code>
     * option in {@link SIXBSWriter}.
     *
     * @see #getValidating()
     */
    public void setValidating(boolean validating) {
        this.validating = validating;
    }

    /**
     * Returns the default sixbs version that should be used for parsing.
     */
    public String getDefaultVersion() {
        return defaultVersion;
    }

    /**
     * Returns the actually used version. May return null if the version has
     * not been parsed/set yet.
     */
    public String getVersion() {
        return version;
    }

    void setVersion(String version) {
        this.version = version;
    }

    /**
     * Called by the ContentHandler.
     */
    void setStillParsing(boolean stillParsing) {
        this.stillParsing = stillParsing;
    }

    /**
     * Sets the number of Objects which are read in advance.
     * If you have very big objects you want to keep this low, in order
     * to keep memory consumption low. If speed is more important, set
     * this to a high value. The default value is 10.
     *
     * @see #getReadAhead()
     */
    public void setReadAhead(int readAhead) {
        this.readAhead = readAhead;
    }

    /**
     * Returns the number of Objects that are read in advance.
     *
     * @see #setReadAhead(int)
     */
    public int getReadAhead() {
        return readAhead;
    }

    /**
     * Starts the reading of objects. Note that internally more than
     * a single object is read. Therefore exceptions which might occur, do
     * not have to stem from the object you are currently trying to read, but
     * may be from an object that is read in advance.
     *
     * @see #setReadAhead(int)
     * @exception EOFException if the end of file is reached.
     */
    public Object readObject() throws IOException {
        handlePendingException();
        checkThread();
        Object obj = null;
        synchronized (lock) {
            try {
                while (objects.size() == 0 && stillParsing) {
                    lock.wait(250);
                    handlePendingException();
                }
            }
            catch (InterruptedException ie) {
                //ie.printStackTrace();
                throw new SIXBSException(ie);
            }

            if (objects.size() > 0) {
                obj = objects.remove(0);
                lock.notify();
            }
            else {
                throw new EOFException();
            }
        }
        return obj;
    }

    /**
     * Adds an object to the queue. Is used by ContentHandlers.
     */
    void addObject(Object obj) {
        synchronized (lock) {
            objects.add(obj);
            lock.notify();
            try {
                // if we already read some objects, better wait a bit, until they are read
                // in order to keep memory consumption low or at least a bit more controllable.
                while (objects.size() > readAhead && stillParsing) {
                    lock.wait(1000);
                }
            }
            catch (InterruptedException ie) {
                ie.printStackTrace();
            }
        }
    }

    /**
     * Indicates whether we can read another object.
     *
     * @return true, if another object can be read.
     */
    public boolean hasNext() {
        try {
            checkThread();
        }
        catch (SIXBSException s) {
            setPendingException(s);
            return false;
        }
        synchronized (lock) {
            try {
                while (objects.size() == 0 && stillParsing) {
                    lock.wait(75);
                }
            }
            catch (InterruptedException ie) {
                return false;
            }
            return objects.size() > 0;
        }
    }

    private void checkThread() throws SIXBSException {
        if (parseThread == null && stillParsing) {
            parseThread = new ParseThread(new VersionedContentHandlerWrapper(this));
            // make sure things don't hang because of this thread.
            parseThread.setDaemon(true);
            parseThread.start();
        }
    }


    /**
     * Closes this reader. It is <i>essential</i> that you close the reader after you used it.
     */
    public void close() {
        if (parseThread != null && stillParsing) {
            setStillParsing(false);
            synchronized (lock) {
                lock.notify();
            }
            // maybe this can be done better... (rik)
            try {
                parseThread.join(1000);
            }
            catch (InterruptedException ie) {
                ie.printStackTrace();
            }
            if (parseThread.isAlive()) {
                parseThread.stop();
            }
        }
        parseThread = null;
        setStillParsing(false);
    }

    private void setPendingException(IOException e) {
        pendingException = e;
    }

    private void handlePendingException() throws IOException {
        if (pendingException != null) {
            throw pendingException;
        }
    }

    protected void finalize() throws Throwable {
        close();
    }

    private static URL findResourceInJar(String sibling, String name) throws ClassNotFoundException {
        ClassLoader loader=Class.forName(sibling).getClassLoader();
        return loader.getResource(name);
    }

    /**
     * Parses the document asynchronously, so that
     * readObject returns immediately, when a new Object has been read.
     */
    private class ParseThread extends Thread {
        private XMLReader xr;
        public ParseThread(ContentHandler ch) throws SIXBSException {
            super("SIXBS ParseThread - " + ch);
            try {
                if (saxParserFactory == null) {
                    saxParserFactory = SAXParserFactory.newInstance();
                    saxParserFactory.setValidating(false);
                }
                // Create a JAXP SAXParser
                SAXParser saxParser = null;
                synchronized (saxParserFactory) {
                    saxParserFactory.setValidating(getValidating());
                    saxParser = saxParserFactory.newSAXParser();
                }
                xr = saxParser.getXMLReader();
                xr.setContentHandler(ch);
                URL dtd=findResourceInJar(getClass().getName(),"com/tagtraum/sixbs/sixbs1.1.dtd");
                EntityResolverImpl resolver = new EntityResolverImpl();
                resolver.put("-//tagtraum industries//DTD SIXBS 1.1//EN", dtd);
                xr.setEntityResolver(resolver);
                xr.setErrorHandler(new ErrorHandlerImpl());
            }
            catch (SAXException se) {
                throw new SIXBSException(se);
            }
            catch (ParserConfigurationException pce) {
                throw new SIXBSException(pce);
            }
            catch (ClassNotFoundException cnfe) {
                throw new SIXBSException(cnfe);
            }
        }

        /**
         * Does the parsing. Exceptions are reported with {@link SIXBSReader#setPendingException(SIXBSException)}
         * and thrown at the next opportunity in {@link SIXBSReader#readObject()}.
         */
        public void run() {
            try {
                xr.parse(inputSource);
            }
            catch (SAXException se) {
                setPendingException(new SIXBSException(se));
            }
            catch (IOException ioe) {
                setPendingException(ioe);
            }
            catch (Throwable t) {
                // shouldn't happen
                t.printStackTrace();
            }
            setStillParsing(false);
        }

    }

    private class ErrorHandlerImpl implements ErrorHandler {
        public void warning(SAXParseException e) throws SAXException {
            System.err.println(e);
        }

        public void error(SAXParseException e) throws SAXException {
            setPendingException(new SIXBSException(e));
        }

        public void fatalError(SAXParseException e) throws SAXException {
            setPendingException(new SIXBSException(e));
        }
    }
}