/*

License $Id: SIXBSWriter.java,v 1.5 2001/08/06 16:46:42 Hendrik Exp $

Copyright (c) 2001 tagtraum industries.

The sixbs library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

For LGPL see <http://www.fsf.org/copyleft/lesser.txt>

We encourage your feedback and suggestions and want to use your feedback to
improve the Software. Send all such feedback to:
feedback@tagtraum.com

For more information on tagtraum industries, Hendrik Schreiber and sixbs
please see <http://www.tagtraum.com/>.

*/

package com.tagtraum.sixbs;

import java.util.ResourceBundle;
import java.io.*;
import java.lang.reflect.*;

/**
 * Serializes beans to XML.
 *
 * @author 	<a href="mailto:hs@tagtraum.com">Hendrik Schreiber</a>
 * @version $Id: SIXBSWriter.java,v 1.5 2001/08/06 16:46:42 Hendrik Exp $
 * @see SIXBSReader
 */
public class SIXBSWriter {

    /**
     * Source-Version
     */
    public static String	vcid =
	"$Id: SIXBSWriter.java,v 1.5 2001/08/06 16:46:42 Hendrik Exp $";

    private static ResourceBundle localStrings = ResourceBundle.getBundle("com.tagtraum.sixbs.localStrings");
    private AbstractSIXBSWriterImpl writer;

    /**
     * Uses default char encoding. The default AdapterFinder is used.
     *
     * @param out OutputStream to write objects to with the system's default encoding.
     * @param append if true, neither the XML declaration (&lt;?xml version="1.0"?&gt;) nor the sixbs tags are written.
     */
    public SIXBSWriter(OutputStream out, boolean append) throws IOException {
        this(new OutputStreamWriter(out), AdapterFinder.getDefaultAdapterFinder(), append);
    }

    /**
     * For performance reasons the OutputStream should be buffered. sixbs does <i>not</i>
     * buffer. The default AdapterFinder is used.
     *
     * @param out OutputStream to write objects to with the system's default encoding.
     * @param append if true, neither the XML declaration (&lt;?xml version="1.0"?&gt;) nor the sixbs tags are written.
     * @param encoding char set to be used
     */
    public SIXBSWriter(OutputStream out, boolean append, String encoding) throws IOException {
        this(new OutputStreamWriter(out, encoding), AdapterFinder.getDefaultAdapterFinder(), append);
    }

    /**
     * For performance reasons the Writer should be buffered. sixbs does <i>not</i>
     * buffer. sixbs will write with the systems default encoding or the encoding
     * set in the provided buffer. No XML declaration is written. The default AdapterFinder
     * is used. Append is set to false.
     *
     * @param out Writer to write objects to.
     */
    public SIXBSWriter(Writer out) throws IOException {
        this(out, AdapterFinder.getDefaultAdapterFinder(), false, null);
    }

    /**
     * For performance reasons the Writer should be buffered. sixbs does <i>not</i>
     * buffer. sixbs will write with the systems default encoding or the encoding
     * set in the provided writer.
     *
     * @param append if true, neither the XML declaration (&lt;?xml version="1.0"?&gt;) nor the sixbs tags are written.
     * @param out Writer to write objects to.
     */
    public SIXBSWriter(Writer out, boolean append) throws IOException {
        this(out, AdapterFinder.getDefaultAdapterFinder(), append, null);
    }

    /**
     * For performance reasons the Writer should be buffered. sixbs does <i>not</i>
     * buffer. sixbs will write with the systems default encoding or the encoding
     * set in the provided writer.
     *
     * @param out Writer to write objects to.
     * @param append if true, neither the XML declaration (&lt;?xml version="1.0"?&gt;) nor the sixbs tags are written.
     */
    public SIXBSWriter(Writer out, AdapterFinder adapterFinder, boolean append) throws IOException {
        this(out, adapterFinder, append, null);
    }

    /**
     * For performance reasons the Writer should be buffered. sixbs does <i>not</i>
     * buffer. sixbs will write with the systems default encoding or the encoding
     * set in the provided writer.
     *
     * @param out Writer to write objects to with the specified encoding.
     * @param append if true, neither the XML declaration (&lt;?xml version="1.0"?&gt;) nor the sixbs tags are written.
     * @param adapterFinder the adapterFinder to use
     */
    public SIXBSWriter(OutputStreamWriter out, AdapterFinder adapterFinder, boolean append) throws IOException {
        this(out, adapterFinder, append, out.getEncoding());
    }

    /**
     * For performance reasons the Writer should be buffered. sixbs does <i>not</i>
     * buffer. sixbs will write with the systems default encoding or the encoding
     * set in the provided writer.
     *
     * @param out Writer to write objects to with the specified encoding.
     * @param append if true, neither the XML declaration (&lt;?xml version="1.0"?&gt;) nor the sixbs tags are written.
     * @param adapterFinder the adapterFinder to use
     * @param encoding encoding that is used by the writer - set null, if unknwon
     */
    public SIXBSWriter(Writer out, AdapterFinder adapterFinder, boolean append, String encoding) throws IOException {
        this(out, adapterFinder, append, encoding, SIXBSReader.DEFAULTVERSION);
    }

    /**
     * For performance reasons the Writer should be buffered. sixbs does <i>not</i>
     * buffer. sixbs will write with the systems default encoding or the encoding
     * set in the provided writer.
     *
     * @param out Writer to write objects to with the specified encoding.
     * @param append if true, neither the XML declaration (&lt;?xml version="1.0"?&gt;) nor the sixbs tags are written.
     * @param adapterFinder the adapterFinder to use
     * @param encoding encoding that is used by the writer - set null, if unknwon
     * @param version version of the writerimpl that is to be instantiated
     */
    public SIXBSWriter(Writer out, AdapterFinder adapterFinder, boolean append, String encoding, String version) throws IOException {
        try {
            Class handlerClass = Class.forName("com.tagtraum.sixbs.SIXBSWriterImpl_" + version.replace('.', '_'));
            Constructor constructor = (Constructor)handlerClass.getConstructor(new Class[] {Writer.class, AdapterFinder.class, Boolean.TYPE, String.class});
            writer = (AbstractSIXBSWriterImpl)constructor.newInstance(new Object[] {out, adapterFinder, new Boolean(append), encoding});
        }
        catch (Exception e) {
            throw new SIXBSException(localStrings.getString("writerimpl_instantiation_failed") + version, e);
        }
    }

    /**
     * Writes an object.
     */
    public void writeObject(Object obj) throws IOException {
        writer.writeObject(obj);
    }

    /**
     * Flush the stream.  If the stream has saved any characters from the
     * various write() methods in a buffer, write them immediately to their
     * intended destination.  Then, if that destination is another character or
     * byte stream, flush it.  Thus one flush() invocation will flush all the
     * buffers in a chain of Writers and OutputStreams.
     *
     * @exception  IOException  If an I/O error occurs
     */
    public void flush() throws IOException {
        writer.flush();
    }

    /**
     * Close the stream, flushing it first.  Once a stream has been closed,
     * further write() or flush() invocations will cause an IOException to be
     * thrown.
     * If this writer does not operate in append-mode, a closing &lt;/sixbs&gt;
     * tag will be written.
     *
     * @exception  IOException  If an I/O error occurs
     */
    public void close() throws IOException {
        writer.close();
    }

}